<?php

namespace Mnv\Admin\Controllers;

use Mnv\Core\Config;
use Mnv\Core\Design;
use Mnv\Core\UserInfo;
use Mnv\Core\Managers;
use Mnv\Core\Managers\Role;
use Mnv\Core\Managers\AdminAuth;
use Mnv\Core\Managers\Errors\AuthError;
use Mnv\Core\Managers\Errors\UsernameRequiredError;
use Mnv\Core\Managers\Exceptions\UserRoleExistsException;
use Mnv\Core\Managers\Exceptions\InvalidPasswordException;
use Mnv\Core\Managers\Exceptions\TooManyRequestsException;
use Mnv\Core\Managers\Exceptions\UnknownUsernameException;
use Mnv\Core\Managers\Exceptions\AttemptCancelledException;
use Mnv\Core\Managers\Exceptions\EmailNotVerifiedException;
use Mnv\Core\Managers\Exceptions\ValidatePasswordException;

/**
 * Class Login
 */
class AuthAdmin
{
    private $authAdmin;
    private $smarty;
    private $managers;
    private $config;

    private $action;
    private $login;
    private $limit_error = 10;

    private $rememberDuration;

    public $errors = array();
    public $messages = array();

    /**
     * Login constructor.
     */
    public function __construct()
    {
        $this->config       = Config::init()->config();
        $this->smarty       = Design::init()->design();

        $this->authAdmin    = new AdminAuth(UserInfo::get_ip(), true, 60 * 5);
        $this->managers     = new Managers($this->authAdmin->getUserId());

        loadLanguage($this->config['admin_language']);

        $this->action     = getRequestVar('action','',  true);
        $this->login      = (object) getRequestVar('login','',  '');

        if ($this->managers->getUserBanned()) $this->numberAttempts($this->authAdmin->getBanned());

        if (isset($this->login->remember) && $this->login->remember == 1) {
            // оставаться в системе в течение одного года
            $this->rememberDuration = (int) (60 * 60 * 24 * 365.25);
        } else {
            // не оставаться в системе после окончания сеанса
            $this->rememberDuration = null;
        }

        if (!empty($this->action)) {
            if ($this->action == 'login') {
                $this->adminLogin($this->authAdmin);
            } else if ($this->action == 'logout') {
                $this->logout($this->authAdmin);
            }
        }

        if (!empty($this->errors))   $this->smarty->assign('errors',   $this->errors);
        if (!empty($this->messages)) $this->smarty->assign('messages', $this->messages);
    }


    /**
     * @param AdminAuth $authAdmin
     * @return array|string
     */
    public function adminLogin(AdminAuth $authAdmin)
    {
        try {
            $this->login->loginName = htmlspecialchars($this->login->loginName);
            $this->login->password  = htmlspecialchars($this->login->password);
            $authAdmin->loginAdminWithUsername($this->login->loginName, $this->login->password, $this->rememberDuration);

            $this->redirectRoleAdmin($authAdmin);

        }  catch (UsernameRequiredError $e) {
            $this->errors[] = lang('login:errors:5'); //lang('login:errors:0');
        } catch (UnknownUsernameException $e) {
            $this->errors[] = 'Вы уверены что знаете имя пользователя?';
        }  catch (ValidatePasswordException $e) {
            $this->errors[] = lang('login:errors:6');
        } catch (InvalidPasswordException $e) {
            if ($this->managers->getUserBanned()) $authAdmin->setBanned();
            $this->errors[] = 'Не пытайтесь подобрать пароль';          // lang('login:errors:3'); //'неправильный пароль';
        } catch (EmailNotVerifiedException $e) {
            $this->errors[] = 'Email не подтвержден';
        } catch (UserRoleExistsException $e) {
            if ($this->managers->getUserBanned()) $authAdmin->setBanned();
            $this->errors[] = 'Нет доступа для этого аккаунта';
        } catch (TooManyRequestsException $e) {
            if ($this->managers->getUserBanned()) $this->managers->setUserBanned();
            $this->errors[] = 'Слишком много запросов';
        } catch (AuthError | AttemptCancelledException $e) {
            $this->errors[] = $e->getMessage();
        }
    }

    /**
     * @param AdminAuth $authAdmin
     */
    private function redirectRoleAdmin(AdminAuth $authAdmin)
    {
        if ($authAdmin->hasAnyRole(Role::ADMIN, Role::DEVELOPER, Role::MANAGER, Role::MODERATOR, Role::EDITOR, Role::WRITER)) {
            redirect('/admin/index.php?mod=main');
        }
    }

    /**
     * Выход
     * @param AdminAuth $authAdmin
     * @throws AuthError
     */
    public function logout(AdminAuth $authAdmin)
    {
        $authAdmin->logOut();
        if (!$authAdmin->check()) {
            redirect('index.php?mod=auth');
        }
    }


    /**
     * Проверка на количество попыток
     * @param $number
     */
    public function numberAttempts($number)
    {
        if (!empty($number)) {
            $limit_cnt = $this->limit_error - $number;
            if ($number >= 0 && $number <= 8) {
                $this->smarty->assign('error_message', 'auth_wrong');
                $this->smarty->assign('limit_cnt', $limit_cnt);
            } else if ($number == 9) {
                $this->smarty->assign('message_limit', "Осталась последняя попытка!<br>Советуем не тратить свое время)");
            } else if ($number == 10) {
                $this->managers->setUserBanned();
                $this->smarty->assign('isBanned', 'banned_message');
            }
        }
    }

    /**
     * @return string
     * @throws \SmartyException
     */
    public function fetch()
    {
//        print_r($_SESSION);
        if ($this->authAdmin->isLoggedIn())
            $this->redirectRoleAdmin($this->authAdmin);

        return $this->smarty->fetch('auth.tpl');
    }

}









