<?php

namespace Mnv\Core;

use Smarty;
use SmartyException;

/**
 * Class Application
 * custom class Smarty
 *
 * @package System
 */
final class Design
{
    public $caching =  0;

    /** @var Smarty  */
    private $smarty;

    /** @var Design */
    private static $instance;

    /** @var bool */
    private $smartySecurity = false;

    /** @var string */
    private $defaultTemplateDir;

    /** @var array */
    private $smartyModifiers = [];

    /** @var array */
    private $smartyFunctions = [];

    /** @return Design */
    public static function init(): Design
    {
        if (!self::$instance) {
            self::$instance = new Design();
        }
        return self::$instance;
    }
    /**
     * Design constructor.
     * @throws SmartyException
     */
    public function __construct()
    {
        $this->smarty = new Smarty();
        $config     = Config::init()->config();

        /**
         * @var  $compile_check
         * При каждом вызове РНР-приложения Smarty проверяет, изменился или нет текущий шаблон с момента последней компиляции.
         * Если шаблон изменился, он перекомпилируется.
         * В случае, если шаблон еще не был скомпилирован, его компиляция производится с игнорированием значения этого параметра.
         * По умолчанию эта переменная установлена в true.
         * В момент, когда приложение начнет работать в реальных условиях (шаблоны больше не будут изменяться),
         * этап проверки компиляции становится ненужным.
         * В этом случае проверьте, чтобы переменная $compile_check была установлена в "false" для достижения максимальной производительности.
         * Учтите, что если вы присвоите этой переменной значение "false", и файл шаблона будет изменен, вы *НЕ* увидите изменений в выводе шаблона до тех пор,
         * пока шаблон не будет перекомпилирован.
         * Если caching и compile_check активированы, файлы кэша будут регенерированы при обновлении связанных с ним шаблонов или конфигурационных файлов
         */
        $this->smarty->compile_check   = Smarty::COMPILECHECK_ON;

//        $this->smarty->caching         = true;
        $this->smarty->auto_literal    = true;
        $this->smarty->debugging       = false;
        $this->smarty->error_reporting = E_ALL & ~E_NOTICE;

        $allowedPhpFunctions = ['escape', 'cat', 'count', 'in_array', 'nl2br', 'str_replace', 'reset', 'floor', 'round', 'ceil', 'max', 'min', 'number_format', 'print_r', 'var_dump', 'printa', 'file_exists', 'stristr', 'strtotime', 'empty', 'urlencode', 'intval', 'isset', 'sizeof', 'is_array', 'array_intersect', 'time', 'array', 'base64_encode', 'implode', 'explode', 'preg_replace', 'preg_match', 'key', 'json_encode', 'json_decode', 'is_file', 'date', 'strip_tags'];

        if ($this->smartySecurity == true) {
            $this->smarty->enableSecurity();
            $this->smarty->security_policy->php_modifiers = $allowedPhpFunctions;
            $this->smarty->security_policy->php_functions = $allowedPhpFunctions;
//            $this->smarty->security_policy->php_handling = Smarty::PHP_REMOVE;
            $this->smarty->security_policy->secure_dir = array(
                GLOBAL_ROOT . 'templates/' . $config['theme'],
                GLOBAL_ROOT . 'admin/templates',
            );
        }

        $tmpPath = sprintf('%s/temp/smarty/', GLOBAL_ROOT);

        $this->defaultTemplateDir = GLOBAL_ROOT.'/themes/'.$config['theme'];
        $this->smarty->setCompileDir($tmpPath . 'compile');
        $this->smarty->setCompileId('main' . SITE_LANG_POSTFIX.  '-' . $config['theme']);
        $this->smarty->setTemplateDir($this->defaultTemplateDir);


        // Создаем папку для скомпилированных шаблонов текущей темы
        if (!is_dir($this->smarty->getCompileDir())) mkdir($this->smarty->getCompileDir(), 0777);

        $this->smarty->setCacheDir($tmpPath . 'cache');

        $this->smarty->addPluginsDir(GLOBAL_ROOT.'/includes/smarty_plugins');
        $this->smarty->default_modifiers = array('escape');

        $this->smarty->registerPlugin('modifier', 'plural',         array($this, 'smarty_modifier_plural'));
        $this->smarty->registerPlugin('modifier', 'upper',          array($this, 'smarty_modifier_upper'));
        $this->smarty->registerPlugin('modifier', 'first',		      array($this, 'first_modifier'));
        $this->smarty->registerPlugin('modifier', 'element',		      array($this, 'fetch_from_array_modifier'));
        $this->smarty->registerPlugin('modifier', 'bold',		          array($this, 'smarty_modifier_bold_words'));
        $this->smarty->registerPlugin('modifier', 'format_time',        array($this, 'smarty_modifier_format_time'));
        $this->smarty->registerPlugin('modifier', 'fsize_format',       array($this, 'smarty_modifier_fsize_format'));
        $this->smarty->registerPlugin('modifier', 'lower',              array($this, 'smarty_modifier_lower'));
        $this->smarty->registerPlugin('modifier', 'nl2br',              array($this, 'smarty_modifier_nl2br'));
        $this->smarty->registerPlugin('modifier', 'string_format',      array($this, 'smarty_modifier_string_format'));
        $this->smarty->registerPlugin('modifier', 'format_phone',       array($this, 'smarty_modifier_format_phone'));
        $this->smarty->registerPlugin('modifier', 'strip_phone',        array($this, 'smarty_modifier_strip_phone'));
        $this->smarty->registerPlugin('modifier', 'strip_html',         array($this, 'smarty_modifier_strip_html'));
        $this->smarty->registerPlugin('modifier', 'strip_tags',         array($this, 'smarty_modifier_strip_tags'));
        $this->smarty->registerPlugin('modifier', 'page_break', 		    array($this, 'page_break_modifier'));

        $this->smarty->registerPlugin('function', 'url', 		        array($this, 'url_modifier'));
        $this->smarty->registerPlugin('function', 'image_file_name_exists',   array($this, 'image_file_name_exists'));


    }

    /** @return Design */
    public function design(): Design
    {
        return $this;
    }

    public function __clone()
    {
        throw new \LogicException('The Design helper cannot be cloned');
    }

    public function __wakeup()
    {
        throw new \LogicException('The Design helper cannot be serialised');
    }

    /**
     * @param string $tag
     * @param string|null $default
     * @return array|string
     */
    public function __(string $tag, ?string $default)
    {
        return __($tag, $default);
    }


    /**
     * @param int $num
     * @param string|null $time
     */
    public function caching(int $num, ?string $time)
    {
        $this->smarty->caching = $num;
        if (!empty($time)) $this->smarty->cache_lifetime = $time;
    }


    /**
     * @param null $template
     * @param null $cache_id
     * @param null $compile_id
     * @param null $parent
     * @return bool
     * @throws SmartyException
     */
    public function isCached($template = null, $cache_id = null, $compile_id = null, $parent = null)
    {
       return $this->smarty->isCached($template, $cache_id, $compile_id, $parent);
    }


    /**
     * Подключение переменной в шаблон
     *
     * @param $tpl_var
     * @param null $value
     * @param bool $nocache
     * @return Smarty
     */
    public function assign($tpl_var, $value = null, bool $nocache = false): Smarty
    {
       return $this->smarty->assign($tpl_var, $value, $nocache);
    }

    /**
     * @throws SmartyException
     */
    private function registerSmartyPlugins()
    {
        foreach ($this->smartyModifiers as $tag => $callback) {
            $this->smarty->registerPlugin('modifier', $tag, $callback);
            unset($this->smartyModifiers[$tag]);
        }

        foreach ($this->smartyFunctions as $tag => $callback) {
            $this->smarty->registerPlugin('function', $tag, $callback);
            unset($this->smartyFunctions[$tag]);
        }
    }

    /**
     * Отображение конкретного шаблона
     *
     * @param null $template
     * @param null $cache_id
     * @param null $compile_id
     * @param null $parent
     * @return false|string
     * @throws SmartyException
     */
    public function fetch($template = null, $cache_id = null, $compile_id = null, $parent = null)
    {
        // Передаем в дизайн то, что может понадобиться в нем
        try {
            $this->registerSmartyPlugins();
        } catch (SmartyException $e) {
            print_r( $e->getMessage());
        }
        return $this->smarty->fetch($template, $cache_id, $compile_id, $parent);
    }

    /**
     * Проверка существует ли данный файл шаблона
     *
     * @param $tplFile
     * @return bool
     * @throws SmartyException
     */
    public function templateExists($tplFile)
    {
        return $this->smarty->templateExists(trim(preg_replace('~[\n\r]*~', '', $tplFile)));
    }

    /**
     * отображает шаблон Smarty
     * @param null $template
     * @throws SmartyException
     */
    public function display($template = null, $cache_id = null, $compile_id = null, $parent = null)
    {
        $this->smarty->display($template, $cache_id, $compile_id, $parent);
    }

    /**
     * Присваивает значения переменным шаблона по ссылке
     *
     * @param $var
     * @param $value
     * @param bool $nocache
     */
    public function assignByRef($var, &$value, bool $nocache = false)
    {
        $this->smarty->assignByRef($var, $value, $nocache);
    }


    /**
     * Установка директории файлов шаблона(отображения)
     * @param $dir
     * @param bool|false $isConfig
     */
    public function setTemplateDir($dir, bool $isConfig = false)
    {
        $this->smarty->setTemplateDir($dir, $isConfig);
    }


    /**
     * Установка директории для готовых файлов для отображения
     * @param $dir
     */
    public function setCompileDir($dir)
    {
        $this->smarty->setCompileDir($dir);
    }

    /**
     * Установить каталог кеша
     * @param $dir
     */
    public function setCacheDir($dir)
    {
        $this->smarty->setCacheDir($dir);
    }

    /**
     * Идентификатор компиляции
     * @param $compileId
     */
    public function setCompileId($compileId)
    {
        $this->smarty->setCompileId($compileId);
    }

    /** Установка директории файлов шаблона(отображения) */
    public function setTemplatesDir($dir)
    {
        $dir = rtrim($dir, '/') . '/';
        if (!is_string($dir)) {
            throw new \Exception("Param \$dir must be string");
        }

        $this->defaultTemplateDir = $dir;
        $this->setSmartyTemplatesDir($dir);
    }

    public function setSmartyTemplatesDir($dir)
    {
        $this->smarty->setTemplateDir($dir);
    }

    /** Выборка переменой */
    public function getTemplateVars(string $name)
    {
        return $this->smarty->getTemplateVars($name);
    }

    /** Очистка кэша Smarty */
    public function clearAssign($var)
    {
        $this->smarty->clearAssign($var);
    }

    /** Очистка кэша Smarty */
    public function clearAllCache()
    {
        $this->smarty->clearAllCache();
    }

    /**
     * @param array $escape
     */
    public function defaultModifiers(array $escape)
    {
        $this->smarty->default_modifiers = $escape;
    }

    /**
     * @param $pluginsDir
     */
    public function addPluginsDir($pluginsDir)
    {
        $this->smarty->addPluginsDir($pluginsDir);
    }

    /**
     * @param string $type
     * @param callable $callback
     * @param null $name
     * @return mixed
     * @throws SmartyException
     */
    public function registerFilter(string $type, callable $callback, $name = null)
    {
        return $this->smarty->registerFilter($type, $callback, $name);
    }

    /**
     * @param $template
     * @param $fileName
     */
    public function registerResource($template, $fileName)
    {
        $this->smarty->registerResource($template, $fileName);
    }

//    /**
//     * @param $block
//     * @param $blockImpl
//     * @param bool $cacheable
//     * @param null $cacheAttrs
//     * @throws SmartyException
//     */
//    public function registerBlock($block, $blockImpl, bool $cacheable = true, $cacheAttrs = null)
//    {
//        $this->smarty->registerBlock($block, $blockImpl, $cacheable, $cacheAttrs);
//    }





    /** ************************        ***************************** */

    /**
     * @param array $params
     * @param $key
     * @param $value
     * @param $as
     * @return false|mixed|null
     */
    public function fetch_from_array_modifier($params = array(), $key, $value, $as)
    {
//        print_r($params);
        if(!is_array($params))
            return false;

        $result = null;
        foreach ($params as $param) {
            if (isset($param->{$key})) {
                if ($param->{$key} != $value) continue;
                $result = $param->{$as};
            }
        }

//        print_r($result);
        return $result;
    }

    /**
     * @param array $params
     * @return false|mixed
     */
    public function first_modifier($params = array())
    {
        if(!is_array($params))
            return false;
        return reset($params);
    }

    /**
     * @param $params
     * @return mixed
     */
    public function url_modifier($params)
    {
        if(is_array(reset($params)))
            return $this->request->url(reset($params));
        else
            return $this->request->url($params);
    }

    /**
     * @param $string
     * @param string $words
     * @param int $limit
     * @return array|mixed|string|string[]|null
     */
    public function smarty_modifier_bold_words($string, $words = '', $limit = -1)
    {
        if (preg_match_all("/\w+/", $words, $matches)) {
            foreach($matches[0] as $word) $string = preg_replace("/\b($word)\b/i", "<b>$1</b>", $string, $limit);
        }
        return $string;
    }

    /**
     * @param $string
     * @param string $format
     * @return false|string|void
     */
    public function smarty_modifier_format_time($string, $format = 'r')
    {
        if ($string != '') {
            return date($format, $string);
        } else {
            return;
        }
    }

    /**
     * @param $size
     * @param string $format
     * @param int $precision
     * @param string $dec_point
     * @param string $thousands_sep
     * @return string
     */
    public function smarty_modifier_fsize_format($size, $format = '', $precision = 2, $dec_point = ".", $thousands_sep = ",")
    {
        $format = strtoupper($format);

        static $sizes = array();

        if(!count($sizes)) {
            $b = 1024;
            $sizes["B"]        =    1;
            $sizes["KB"]    =    $sizes["B"]  * $b;
            $sizes["MB"]    =    $sizes["KB"] * $b;
            $sizes["GB"]    =    $sizes["MB"] * $b;
            $sizes["TB"]    =    $sizes["GB"] * $b;

            $sizes = array_reverse($sizes,true);
        }

        //~ get "human" filesize
        foreach($sizes AS $unit => $bytes) {
            if($size > $bytes || $unit == $format) {
                //~ return formatted size
                return    number_format($size / $bytes, $precision, $dec_point, $thousands_sep)." ".$unit;
            } //~ end if
        } //~ end foreach
    }

    /**
     * @param $string
     * @return string
     */
    public function smarty_modifier_upper($string)
    {
        return strtoupper($string);
    }

    /**
     * @param $string
     * @return string
     */
    public function smarty_modifier_lower($string)
    {
        return strtolower($string);
    }

    /**
     * @param $string
     * @param $format
     * @return string
     */
    public function smarty_modifier_string_format($string, $format)
    {
        return sprintf($format, $string);
    }

    /**
     * @param $phone
     * @return array|string|string[]|null
     */
    public function smarty_modifier_format_phone($phone)
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        return preg_replace('/([0-9]{3})([0-9]{2})([0-9]{3})([0-9]{2})([0-9]{2})/', '+$1 ($2) $3 $4 $5', $phone);
    }

    /**
     * @param $phone
     * @return string
     */
    public function smarty_modifier_strip_phone($phone)
    {
        $phone = preg_replace('/[^0-9]/', '', $phone);
        return '+' . $phone;
    }

    /**
     * @param $number
     * @param $singular
     * @param $plural1
     * @param null $plural2
     * @return mixed
     */
    public function smarty_modifier_plural($number, $singular, $plural1, $plural2 = null)
    {
        $number = abs($number);
        if(!empty($plural2)) {
            $p1 = $number%10;
            $p2 = $number%100;
            if ($number == 0) {
                return $plural1;
            }
            if ($p1 == 1 && !($p2 >= 11 && $p2 <= 19)) {
                return $singular;
            } elseif ($p1 >= 2 && $p1 <= 4 && !($p2 >= 11 && $p2 <= 19)) {
                return $plural2;
            } else {
                return $plural1;
            }
        } else {
            if ($number == 1) {
                return $singular;
            } else {
                return $plural1;
            }
        }
    }

    /**
     * @param $string
     * @return string
     */
    public function smarty_modifier_nl2br($string)
    {
        return nl2br($string);
    }

    /**
     * @param $string
     * @param bool $replace_with_space
     * @return array|string|string[]|null
     */
    public function smarty_modifier_strip_tags($string, $replace_with_space = true)
    {
        if ($replace_with_space)
            return preg_replace('!<[^>]*?>!', ' ', $string);
        else
            return strip_tags($string);
    }

    /**
     * @param $string
     * @return array|string|string[]|null
     */
    public function smarty_modifier_strip_html($string)
    {
        $search = array ("'<script[^>]*?>.*?</script>'si", "'<[\/\!]*?[^<>]*?>'si", "'([\r\n])[\s]+'", "'&(quot|#34);'i", "'&(amp|#38);'i", "'&(lt|#60);'i", "'&(gt|#62);'i", "'&(nbsp|#160);'i", "'&(iexcl|#161);'i", "'&(cent|#162);'i", "'&(pound|#163);'i", "'&(copy|#169);'i");
        $replace = array ("", " ", "\\1", "\"", "&", "<", ">", " ", chr(161), chr(162), chr(163), chr(169));
        return preg_replace($search, $replace, $string);
    }

    /**
     * @param $params
     * @return string|null
     */
    public function image_file_name_exists($params): ?string
    {
        $filePath   = fetch_getParam('filePath', $params);
        $class      = fetch_getParam('class', $params);
        $style      = fetch_getParam('style', $params);

        $fileRoot = GLOBAL_ROOT . $filePath;
        if (file_exists($fileRoot)) {
            return '<img src="'. GLOBAL_URL. $filePath . '" class="'.$class.'" style="' . $style . '"/>' ;
        } else {
            return null;
        }
    }

    /**
     * @param $content
     * @param null $key
     * @return array|false|mixed|string|string[]
     */
    public function page_break_modifier($content, $key = NULL)
    {
        if (empty($content)) return false;

        if (stristr($content, '{PAGEBREAK}') !== false) {
            $contentNotParse = $content;
//            var_dump($key);
            if (!is_null($key)) {

                $content = explode('{PAGEBREAK}', $content);
//
                if (isset($content[$key])) {
//                    print_r($content[$key]);
                    return $content[$key];
                } else {
                    return str_replace('{PAGEBREAK}', '', $contentNotParse);
                }
            } else {
                return str_replace('{PAGEBREAK}', '', $content);
            }

        } else {

            return $content;
        }

    }

}