<?php

declare(strict_types=1);

namespace Mnv\Core;

use Mnv\Core\Managers\Role;
use Mnv\Core\Managers\AdminManager;
use Mnv\Core\Files\Image\ImageSizes;

/**
 * Class Managers
 * @package Mnv\Core
 */
final class Managers
{
    /** @var $manager */
    public $manager;

    /** @var int $userId */
    private $userId = 0;

    /**
     * Managers constructor.
     * @param int|null $managerId
     */
    public function __construct(?int $managerId)
    {
        if (!is_null($managerId))
            $this->userId = $managerId;
    }

    /**
     * Получение менеджера
     * @method getAdminManager
     */
    public function getAdminManager(): Managers
    {
        $fields = 'userId, fullName, firstName, middleName, lastName, loginName, accessLevel, permissions, fileId, lang';
        if ($this->manager = DB::init()->connect()->table('users')->select($fields)->where('userId', $this->userId)->where('accessLevel', '<>', Role::CONSUMER)->get()) {
            if (!empty($this->manager->fileId) && $this->manager->fileId > 0) {
                $this->manager->image = $this->getImage($this->manager);
            }

//            $this->manager->accessLevelName = lang('managers:accessRole:' . $this->manager->accessLevel);

            if (!is_null($this->manager->permissions)) {
                $this->permissions();
            }
        }

        return $this;
    }

    /**
     * Получение фото менеджера
     * @param $manager
     * @return ImageSizes
     */
    private function getImage($manager): ?ImageSizes
    {
        $image = null;
        if (!empty($manager->userId)) {
            if ($file = DB::init()->connect()->table('files')->select('fileId, folder, path, fileName, size')->where('fileId', $manager->fileId)->get()) {
                $image = ImageSizes::init()->get(null, $file);
            }
        }
        return $image;
    }

    private function permissions()
    {
        $this->manager->permissions = explode(',', $this->manager->permissions);
        $this->manager->permissions = array_combine($this->manager->permissions, $this->manager->permissions);
    }

    /**
     * проверяем разрешение на доступ к контроллеру
     *
     * @param $permission
     * @return bool
     */
    private function access($permission)
    {
        if (empty($permission)) {
            return false;
        }

        if (is_array($this->manager->permissions))
            return in_array($permission, $this->manager->permissions);


        return false;
    }

    /**
     * Формирование меню и получение меню админ панели
     * stats,overview,visitors,visitor,referer,bots,bot,order,orders,users,sections,section,sliders,slider,catalog,brands,brand,contents,articles,article,products,product,features,feature,news,new,logo,settings,socials,social,maps,languages,language,currencies,currency,robots,error,errors,logaction,lowerrors,syserrors,serverinfo,serverinfo,inivars,managers,manager,templates,filemanagers,filemanager,files,translations,comments,comment,additionally,about,doc,payments,payment,partners,partner,reviews,review,vacancy,vacancys
     */
    public function getMenu()
    {
        $menu = json_decode(file_read(GLOBAL_ROOT."/admin/menu/menu.json"), true);
        // проверяем на пустоту прав, так как у пользователя сайта нет разрешение на доступ к админ панели
        if (!empty($this->manager->permissions)) {
            foreach ($menu as $key => $item) {
                $menu[$key]['children'] = $this->generateChildrenMenu($item['children']);
                if (empty($menu[$key]['children']))
                    unset($menu[$key]);
            }
        }  else {
            foreach ($menu as $key => $item) {
                $menu[$key]['children'] = $this->generateChildrenMenu($item['children']);
            }
        }

//        print_r($menu);
        return $menu;
    }

    /**
     * @param $children
     * @return mixed$this->menu
     */
    private function generateChildrenMenu($children)
    {
        if (!empty($this->manager->permissions)) {
            foreach ($children as $k => $child) {
                if ($this->access($child['fileName'])) {
                    continue;
                } else {
                    unset($children[$k]);
                }
            }
        } else {
            foreach ($children as $k => $child)
                $children[$k] = $child;
        }
        return $children;
    }

    public function getManagerAccessLevel()
    {
        return $this->manager->accessLevel;
    }


    /**
     * Блокировка пользователя или злоумышленника на 10 часов
     */
    public function setUserBanned()
    {
        if (Config::getValue('check_banned_ips') && (Config::getValue('check_banned_ips') == 1)) {
            $ban['visitorIp']   = UserInfo::get_ip();
            $ban['bannedOn']    = date('Y-m-d H:i:s');
            $ban['expiresOn']   = date('Y-m-d H:i:s', strtotime('+10 hours'));

            DB::init()->connect()->table('users_banned_ips')->insert($ban);

            $this->unsetBanned();
        }
    }

    public function getUserBanned(): bool
    {
        if (!DB::init()->connect()->table('users_banned_ips')->where('visitorIp', UserInfo::get_ip())->get()) return true;

        return false;
    }

    /**
     * @return int
     */
    public function unsetBanned(): int
    {
        return $_SESSION['admin'][AdminManager::SESSION_FIELD_BANNED] = 0;
    }
}