<?php

namespace Mnv\Core\Managers;

use Mnv\Core\Database\Throwable\Error;
use Mnv\Core\DB;
use Mnv\Core\Managers\Errors\AuthError;
use Mnv\Core\Managers\Errors\DatabaseError;
use Mnv\Core\Managers\Exceptions\UnknownIdException;
use Mnv\Core\Managers\Exceptions\InvalidEmailException;
use Mnv\Core\Managers\Exceptions\UserRoleExistsException;
use Mnv\Core\Managers\Exceptions\UnknownUsernameException;
use Mnv\Core\Managers\Exceptions\InvalidPasswordException;
use Mnv\Core\Managers\Exceptions\EmailNotVerifiedException;
use Mnv\Core\Managers\Exceptions\AmbiguousUsernameException;
use Mnv\Core\Managers\Exceptions\DuplicateUsernameException;
use Mnv\Core\Managers\Exceptions\UserAlreadyExistsException;


/** Компонент, который может использоваться для административных задач привилегированными и авторизованными пользователями */
final class Administration extends AdminManager {

    /**
     * Administration constructor.
     */
	public function __construct()
    {
	}

	/**
	 * Создает нового user
	 *
	 * @param string $email the email address to register
	 * @param string $password the password for the new account
	 * @param string|null $loginName (optional) the loginName that will be displayed
	 * @return int the ID of the user that has been created (if any)
	 * @throws InvalidEmailException if the email address was invalid
	 * @throws InvalidPasswordException if the password was invalid
	 * @throws UserAlreadyExistsException if a user with the specified email address already exists
	 * @throws AuthError if an internal problem occurred (do *not* catch)
	 */
	public function createUser($email, $password, $loginName = null)
    {
		return $this->createUserInternal(false, $email, $password, $loginName, null);
	}

	/**
     * Создает нового user, обеспечивая при этом уникальность имени пользователя.
	 * Creates a new user while ensuring that the username is unique
	 *
	 * @param string $email the email address to register
	 * @param string $password the password for the new account
	 * @param string|null $loginName (optional) the username that will be displayed
	 * @return int the ID of the user that has been created (if any)
	 * @throws InvalidEmailException if the email address was invalid
	 * @throws InvalidPasswordException if the password was invalid
	 * @throws UserAlreadyExistsException if a user with the specified email address already exists
	 * @throws DuplicateUsernameException if the specified username wasn't unique
	 * @throws AuthError if an internal problem occurred (do *not* catch)
	 */
	public function createUserWithUniqueUsername($email, $password, $loginName = null)
    {
		return $this->createUserInternal(true, $email, $password, $loginName, null);
	}

    /**
     * Удаляет пользователя с указанным ID
     *
     * Это действие не может быть отменено
     *
     * @param int $id ID пользователя для удаления
     * @throws UnknownIdException пользователь с указанным ID не найден
     * @throws AuthError если возникла внутренняя проблема (do *not* catch)
     */
	public function deleteUserById($id)
    {
		$numberOfDeletedUsers = $this->deleteUsersByColumnValue('userId', (int) $id);

		if ($numberOfDeletedUsers === 0) {
			throw new UnknownIdException();
		}
	}

	/**
     * Удаляет пользователя с указанным email address
	 * Это действие не может быть отменено
	 *
	 * @param string $email адрес электронной почты пользователя для удаления
	 * @throws InvalidEmailException если пользователь с указанным адресом электронной почты не найден
	 * @throws AuthError если возникла внутренняя проблема (do *not* catch)
	 */
	public function deleteUserByEmail($email)
    {
		$email = self::validateEmailAddress($email);

		$numberOfDeletedUsers = $this->deleteUsersByColumnValue('email', $email);

		if ($numberOfDeletedUsers === 0) {
			throw new InvalidEmailException();
		}
	}

	/**
     * Удаляет пользователя с указанным `loginName`
	 * Это действие не может быть отменено
	 *
	 * @param string $loginName loginName для входа в систему для удаления
	 * @throws UnknownUsernameException если пользователь с указанным `loginName` не найден
	 * @throws AmbiguousUsernameException если найдено несколько пользователей с указанным именем пользователя
	 * @throws AuthError если возникла внутренняя проблема (do *not* catch)
	 */
	public function deleteUserByUsername($loginName)
    {
		$userData = $this->getUserDataByUsername(\trim($loginName), ['userId'], [Role::ADMIN, Role::DEVELOPER, Role::MANAGER, Role::MODERATOR, Role::EDITOR, Role::WRITER]);

		$this->deleteUsersByColumnValue('userId', (int) $userData['userId']);
	}

	/**
     * Назначает указанную роль пользователю с данным ID
	 * Пользователь может иметь любое количество ролей  (i.e. no role at all, a single role, or any combination of roles)
	 *
	 * @param int $userId ID, которому нужно назначить роль
	 * @param int $role роль в качестве одной из констант из {@see Role} class
	 * @throws UnknownIdException если пользователь с указанным идентификатором не найден
	 *
	 * @see Role
	 */
	public function addRoleForUserById($userId, $role)
    {
		$userFound = $this->addRoleForUserByColumnValue('userId', (int) $userId, $role);

		if ($userFound === false) {
			throw new UnknownIdException();
		}
	}

	/**
     * Назначает указанную роль пользователю с заданным email address
	 * Пользователь может иметь любое количество ролей (i.e. no role at all, a single role, or any combination of roles)
	 *
	 * @param string $userEmail адрес электронной почты пользователя, которому будет назначена роль
	 * @param int $role роль в качестве одной из констант из {@see Role} class
	 * @throws InvalidEmailException если пользователь с указанным адресом электронной почты не найден
	 *
	 * @see Role
	 */
	public function addRoleForUserByEmail($userEmail, $role)
    {
		$userEmail = self::validateEmailAddress($userEmail);

		$userFound = $this->addRoleForUserByColumnValue('email', $userEmail, $role);

		if ($userFound === false) {
			throw new InvalidEmailException();
		}
	}

	/**
	 * Назначает указанную роль пользователю с указанным именем пользователя
	 * Пользователь может иметь любое количество ролей (i.e. no role at all, a single role, or any combination of roles)
	 *
	 * @param string $loginName loginName для входа в систему, которому будет назначена роль
	 * @param int $role роль в качестве одной из констант из {@see Role} class
	 * @throws UnknownUsernameException если пользователь с указанным именем пользователя не найден
	 * @throws AmbiguousUsernameException если найдено несколько пользователей с указанным именем пользователя
	 *
	 * @see Role
	 */
	public function addRoleForUserByUsername($loginName, $role)
    {
		$userData = $this->getUserDataByUsername(\trim($loginName), ['userId'], [Role::ADMIN, Role::DEVELOPER, Role::MANAGER, Role::MODERATOR, Role::EDITOR, Role::WRITER]);

		$this->addRoleForUserByColumnValue('userId', (int) $userData['userId'], $role);
	}

	/**
	 * Удалить указанную роль у пользователя с заданным ID
	 * Пользователь может иметь любое количество ролей (i.e. no role at all, a single role, or any combination of roles)
	 *
	 * @param int $userId ID, у которого нужно удалить роль
	 * @param int $role роль в качестве одной из констант из {@see Role} class
	 * @throws UnknownIdException если пользователь с указанным ID не найден
	 *
	 * @see Role
	 */
	public function removeRoleForUserById($userId, $role)
    {
		$userFound = $this->removeRoleForUserByColumnValue('userId', (int) $userId, $role);

		if ($userFound === false) {
			throw new UnknownIdException();
		}
	}

	/**
	 * Takes away the specified role from the user with the given email address
	 *
	 * A user may have any number of roles (i.e. no role at all, a single role, or any combination of roles)
	 *
	 * @param string $userEmail the email address of the user to take the role away from
	 * @param int $role the role as one of the constants from the {@see Role} class
	 * @throws InvalidEmailException if no user with the specified email address has been found
	 *
	 * @see Role
	 */
	public function removeRoleForUserByEmail($userEmail, $role)
    {
		$userEmail = self::validateEmailAddress($userEmail);

		$userFound = $this->removeRoleForUserByColumnValue('email', $userEmail, $role);

		if ($userFound === false) {
			throw new InvalidEmailException();
		}
	}

	/**
	 * Забирает указанную роль у пользователя с заданным именем пользователя
	 *
	 * У пользователя может быть любое количество ролей (т. е. вообще никакой роли, одна роль или любая комбинация ролей).
	 *
	 * @param string $loginName the loginName of the user to take the role away from
	 * @param int $role the role as one of the constants from the {@see Role} class
	 * @throws UnknownUsernameException if no user with the specified username has been found
	 * @throws AmbiguousUsernameException if multiple users with the specified username have been found
	 *
	 * @see Role
	 */
	public function removeRoleForUserByUsername($loginName, $role)
    {
		$userData = $this->getUserDataByUsername(\trim($loginName), ['userId'], [Role::ADMIN, Role::DEVELOPER, Role::MANAGER, Role::MODERATOR, Role::EDITOR, Role::WRITER]);

		$this->removeRoleForUserByColumnValue('userId', (int) $userData['userId'], $role);
	}

	/**
	 * Возвращает, имеет ли пользователь с данным идентификатором указанную роль
	 *
	 * @param int $userId the ID of the user to check the roles for
	 * @param int $role the role as one of the constants from the {@see Role} class
	 * @return bool
	 * @throws UnknownIdException if no user with the specified ID has been found
	 *
	 * @see Role
	 */
	public function doesUserHaveRole($userId, $role)
    {
		if (empty($role) || !\is_numeric($role)) {
			return false;
		}

		$userId = (int) $userId;
        $rolesBitmask = DB::init()->connect()->table('users')->select('accessLevel')->where('userId', $userId)->get();

		if ($rolesBitmask->accessLevel === null) {
			throw new UnknownIdException();
		}

		$role = (int) $role;
//var_dump(($rolesBitmask->accessLevel & $role) === $role);
		return ($rolesBitmask->accessLevel & $role) === $role;
	}

	/**
	 * Возвращает роли пользователя с заданным ID, сопоставляя числовые значения с их описательными именами.
	 *
	 * @param int $userId the ID of the user to return the roles for
	 * @return array
	 * @throws UnknownIdException if no user with the specified ID has been found
	 *
	 * @see Role
	 */
	public function getRolesForUserById($userId)
    {
        $rolesBitmask = DB::init()->connect()->table('users')->select('accessLevel')->where('userId', '=', (int) $userId)->getValue();

		if ($rolesBitmask === null) {
			throw new UnknownIdException();
		}

		return \array_filter(
			Role::getMap(),
			function ($each) use ($rolesBitmask) {
				return ($rolesBitmask & $each) === $each;
			},
			\ARRAY_FILTER_USE_KEY
		);
	}

	/**
	 * Выполняет вход как пользователь с указанным ID
	 *
	 * @param int $id the ID of the user to sign in as
	 * @throws UnknownIdException if no user with the specified ID has been found
	 * @throws EmailNotVerifiedException if the user has not verified their email address via a confirmation method yet
	 * @throws AuthError if an internal problem occurred (do *not* catch)
	 */
	public function logInAsUserById($id)
    {
		$numberOfMatchedUsers = $this->logInAsUserByColumnValue('userId', (int) $id);

		if ($numberOfMatchedUsers === 0) {
			throw new UnknownIdException();
		}
	}

	/**
     * Войдите в систему как пользователь с указанным `email`
	 *
	 * @param string $email the email address of the user to sign in as
	 * @throws InvalidEmailException if no user with the specified email address has been found
	 * @throws EmailNotVerifiedException if the user has not verified their email address via a confirmation method yet
	 * @throws AuthError if an internal problem occurred (do *not* catch)
	 */
	public function logInAsUserByEmail($email)
    {
		$email = self::validateEmailAddress($email);

		$numberOfMatchedUsers = $this->logInAsUserByColumnValue('email', $email);

		if ($numberOfMatchedUsers === 0) {
			throw new InvalidEmailException();
		}
	}

	/**
     * Войдите в систему как пользователь с указанным отображаемым `loginName`
	 *
	 * @param string $loginName the display name of the user to sign in as
	 * @throws UnknownUsernameException if no user with the specified username has been found
	 * @throws AmbiguousUsernameException if multiple users with the specified username have been found
	 * @throws EmailNotVerifiedException if the user has not verified their email address via a confirmation method yet
	 * @throws AuthError if an internal problem occurred (do *not* catch)
	 */
	public function logInAsUserByUsername($loginName)
    {
		$numberOfMatchedUsers = $this->logInAsUserByColumnValue('loginName', \trim($loginName));

		if ($numberOfMatchedUsers === 0) {
			throw new UnknownUsernameException();
		}
		elseif ($numberOfMatchedUsers > 1) {
			throw new AmbiguousUsernameException();
		}
	}

	/**
	 * Изменяет пароль для пользователя с заданным ID
	 *
	 * @param int $userId ID пользователя, пароль которого необходимо изменить
	 * @param string $newPassword новый пароль для установки
	 * @throws UnknownIdException если пользователь с указанным идентификатором не найден
	 * @throws InvalidPasswordException если желаемый новый пароль был неверным
	 * @throws AuthError если возникла внутренняя проблема (do *not* catch)
	 */
	public function changePasswordForUserById($userId, $newPassword)
    {
		$userId = (int) $userId;
		$newPassword = self::validatePassword($newPassword);

		$this->updatePasswordInternal($userId, $newPassword);

		$this->forceLogoutForUserById($userId);
	}

	/**
	 * Изменяет пароль для пользователя с указанным именем пользователя
	 *
	 * @param string $loginName логин пользователя, пароль которого необходимо изменить
	 * @param string $newPassword новый пароль для установки
	 * @throws UnknownUsernameException если пользователь с указанным именем пользователя не найден
	 * @throws AmbiguousUsernameException если найдено несколько пользователей с указанным именем пользователя
	 * @throws InvalidPasswordException если желаемый новый пароль был неверным
	 * @throws AuthError если возникла внутренняя проблема (do *not* catch)
	 */
	public function changePasswordForUserByUsername($loginName, string $newPassword) {
		$userData = $this->getUserDataByUsername(\trim($loginName), ['userId'], [Role::DEVELOPER, Role::MANAGER, Role::ADMIN, Role::MODERATOR, Role::EDITOR, Role::WRITER]);

		$this->changePasswordForUserById((int) $userData['userId'], $newPassword);
	}

	/**
	 * Удаляет всех существующих пользователей, у которых столбец с указанным именем имеет заданное значение
	 * Вы никогда не должны передавать ненадежные входные данные параметру, который принимает имя столбца
	 *
	 * @param string $columnName имя столбца для фильтрации по filter by
	 * @param mixed $columnValue значение, которое нужно искать в выбранном столбце
	 * @return int the number of deleted users
	 * @throws AuthError если возникла внутренняя проблема (do *not* catch)
	 */
	private function deleteUsersByColumnValue($columnName, $columnValue)
    {
		try {
            return DB::init()->connect()->table('users')->where($columnName, '=', $columnValue)->delete();
		}
		catch (Error $e) {
			throw new DatabaseError($e->getMessage());
		}
	}

	/**
     * Изменяет роли для пользователя, в котором столбец с указанным именем имеет заданное значение.
	 * Вы никогда не должны передавать ненадежные входные данные параметру, который принимает имя столбца
	 *
	 * @param string $columnName the name of the column to filter by
	 * @param mixed $columnValue the value to look for in the selected column
	 * @param callable $modification the modification to apply to the existing bitmask of roles
	 * @return bool whether any user with the given column constraints has been found
	 * @throws AuthError if an internal problem occurred (do *not* catch)
	 *
	 * @see Role
	 */
	private function modifyRolesForUserByColumnValue($columnName, $columnValue, callable $modification)
    {
		try {
            $userData = DB::init()->connect()->table('users')->select('userId, accessLevel')->where($columnName, $columnValue)->get();
		}
		catch (Error $e) {
			throw new DatabaseError($e->getMessage());
		}

		if ($userData === null) {
			return false;
		}

		$newRolesBitmask = $modification($userData['accessLevel']);

		try {
            DB::init()->connect()->table('users')->where('accessLevel', '=', $userData['accessLevel'])->where('userId', '=', (int) $userData['userId'])->update(['accessLevel' => $newRolesBitmask]);

			return true;
		}
		catch (Error $e) {
			throw new DatabaseError($e->getMessage());
		}
	}

	/**
     * Назначает указанную роль пользователю, где столбец с указанным именем имеет заданное значение
	 * Вы никогда не должны передавать ненадежные входные данные параметру, который принимает имя столбца
	 *
	 * @param string $columnName the name of the column to filter by
	 * @param mixed $columnValue the value to look for in the selected column
	 * @param int $role the role as one of the constants from the {@see Role} class
	 * @return bool whether any user with the given column constraints has been found
	 *
	 * @throws AuthError
	 *@see Role
     */
	private function addRoleForUserByColumnValue($columnName, $columnValue, $role)
    {
		$role = (int) $role;

		return $this->modifyRolesForUserByColumnValue(
			$columnName,
			$columnValue,
			function ($oldRolesBitmask) use ($role) {
				return $oldRolesBitmask | $role;
			}
		);
	}

	/**
     * Удаляем указанную роль у пользователя, где столбец с указанным именем имеет заданное значение
	 * Вы никогда не должны передавать ненадежные входные данные параметру, который принимает имя столбца
	 *
	 * @param string $columnName имя столбца для фильтрации по filter by
	 * @param mixed $columnValue значение, которое нужно искать в выбранном столбце
	 * @param int $role роль в качестве одной из констант из {@see Role} class
	 * @return bool был ли найден какой-либо пользователь с заданными ограничениями столбца
	 *
	 * @throws AuthError
	 * @see Role
     */
	private function removeRoleForUserByColumnValue($columnName, $columnValue, $role) {
		$role = (int) $role;

		return $this->modifyRolesForUserByColumnValue($columnName, $columnValue, function ($oldRolesBitmask) use ($role) {
		    return $oldRolesBitmask & ~$role;
		});
	}

	/**
     * Выполняет вход как пользователь, для которого столбец с указанным именем имеет заданное значение
     * Вы никогда не должны передавать ненадежный ввод в параметр, который принимает имя столбца.
	 *
	 * @param string $columnName the name of the column to filter by
	 * @param mixed $columnValue the value to look for in the selected column
	 * @return int the number of matched users (where only a value of one means that the login may have been successful)
	 * @throws EmailNotVerifiedException if the user has not verified their email address via a confirmation method yet
	 * @throws AuthError if an internal problem occurred (do *not* catch)
	 */
	private function logInAsUserByColumnValue($columnName, $columnValue) {
		try {
            $user = DB::init()->connect()->table('users')->select('verified, userId, email, loginName, status, accessLevel')->where($columnName, $columnValue)->get();
		}
		catch (Error $e) {
			throw new DatabaseError($e->getMessage());
		}

		if ((int) $user['verified'] === 1) {
		    $this->onLoginSuccessful($user['userId'], $user['email'], $user['loginName'], $user['status'], $user['accessLevel'], \PHP_INT_MAX, false);
		}
		else {
		    throw new EmailNotVerifiedException();
		}


		return $user;
	}

}
