<?php


namespace Mnv\Core;

/**
 * Class SecurityService
 * @package Mnv\Core
 */
class SecurityService
{
    /** @var string $formTokenLabel */
    private $formTokenLabel = 'csrf-token-label';
    /** @var string $sessionTokenLabel */
    private $sessionTokenLabel = 'CSRF_TOKEN_SESS_ID';

    /** @var array $post */
    private $post = [];

    /** @var array $session */
    private $session = [];

    /** @var array $server */
    private $server = [];

    /** @var array $excludeUrl */
    private $excludeUrl = [];

    /** @var string $hashAlgo */
    private $hashAlgo = 'sha256';

    /** @var bool $hmac_ip */
    private $hmac_ip = true;

    /** @var string $hmacData */
    private $hmacData = 'ABCeNBHVe3kmAqvU2s7yyuJSF2gpxKLC';

    /**
     * SecurityService constructor.
     *
     * @param null $excludeUrl
     * @param array|null $post
     * @param array|null $session
     * @param array|null $server
     */
    public function __construct($excludeUrl = null, array &$post = null, array &$session = null, array &$server = null)
    {
        if (!is_null($excludeUrl)) {
            $this->excludeUrl = $excludeUrl;
        }
        if (!is_null($post)) {
            $this->post = & $post;
        } else {
            $this->post = & $_POST;
        }

        if (!is_null($server)) {
            $this->server = & $server;
        } else {
            $this->server = & $_SERVER;
        }

        if (!is_null($session)) {
            $this->session = & $session;
        } else if (isset($_SESSION) && !is_null($_SESSION)) {
            $this->session = & $_SESSION;
        } else {
            throw new \Error('No session available for persistence');
        }
    }

    /**
     * Вставьте токен CSRF в форму
     */

    public function insertHiddenToken()
    {
        $csrfToken = $this->getCSRFToken();

        echo "<!--\n--><input type=\"hidden\"" . " name=\"" . $this->xsSafe($this->formTokenLabel) . "\"" . " value=\"" . $this->xsSafe($csrfToken) . "\"" . " />";
    }

    /**
     * @return array
     */
    public function getToken(): array
    {
        $csrfToken = $this->getCSRFToken();

        return [
            'name'  => $this->xsSafe($this->formTokenLabel),
            'value' => $this->xsSafe($csrfToken)
        ];
    }

    // xss mitigation functions
    public function xsSafe($data, $encoding = 'UTF-8'): string
    {
        return htmlspecialchars($data, ENT_QUOTES | ENT_HTML401, $encoding);
    }

    /**
     * Генерируйте, сохраняйте и возвращайте токен CSRF
     *
     * @return string[]
     */
    public function getCSRFToken()
    {
        if (empty($this->session[$this->sessionTokenLabel])) {
            $this->session[$this->sessionTokenLabel] = bin2hex(openssl_random_pseudo_bytes(32));
        }

        if ($this->hmac_ip !== false) {
            $token = $this->hMacWithIp($this->session[$this->sessionTokenLabel]);
        } else {
            $token = $this->session[$this->sessionTokenLabel];
        }
        return $token;
    }

    /**
     * Хеширование с удаленным IP-адресом для упрощения соблюдения GDPR и используется hmacData.
     *
     * @param string $token
     * @return string hashed data
     */
    private function hMacWithIp(string $token)
    {
        return \hash_hmac($this->hashAlgo, $this->hmacData, $token);
    }

    /**
     * возвращает текущий URL-адрес запроса
     *
     * @return string
     */
    private function getCurrentRequestUrl(): string
    {
        $protocol = "http";
        if (isset($this->server['HTTPS'])) {
            $protocol = "https";
        }
        return $protocol . "://" . $this->server['HTTP_HOST'] . $this->server['REQUEST_URI'];
    }

    /**
     * основная функция, которая проверяет попытку CSRF.
     *
     * @throws \Exception
     */
    public function validate(): bool
    {
        $currentUrl = $this->getCurrentRequestUrl();
        if (! in_array($currentUrl, $this->excludeUrl)) {
            if (!empty($this->post)) {
                $isAntiCSRF = $this->validateRequest();
                if (! $isAntiCSRF) {
                    // Попытка атаки CSRF
                    // Обнаружена попытка CSRF. Не нужно раскрывать эту информацию для злоумышленников, так что просто терпит неудачу без информации.
                    // Код ошибки 1837 расшифровывается как попытка CSRF, и это наши идентификационные цели.
                    // создать новое \Исключение("Критическая ошибка! Код ошибки: 1837".);
                    throw new \Exception("ERROR! Error Code: 1837");
//                    return false;
                }
                return true;
            }
        }
    }

    /**
     * Фактическая проверка CSRF происходит здесь и возвращает логическое значение
     *
     * @return boolean
     */
    public function isValidRequest(): bool
    {
        $isValid = false;
        $currentUrl = $this->getCurrentRequestUrl();
        if (!in_array($currentUrl, $this->excludeUrl)) {
            if (! empty($this->post)) {
                $isValid = $this->validateRequest();
            }
        }

        return $isValid;
    }

    /**
     * Проверка запроса на основе `session`
     *
     * @return bool
     */
    public function validateRequest(): bool
    {
        if (! isset($this->session[$this->sessionTokenLabel])) {
            // CSRF Token not found
            return false;
        }

        if (! empty($this->post[$this->formTokenLabel])) {
            // Let's pull the POST data
            $token = $this->post[$this->formTokenLabel];
        } else {
            return false;
        }

        if (! \is_string($token)) {
            return false;
        }

        // Grab the stored token
        if ($this->hmac_ip !== false) {
            $expected = $this->hMacWithIp($this->session[$this->sessionTokenLabel]);
        } else {
            $expected = $this->session[$this->sessionTokenLabel];
        }

        return \hash_equals($token, $expected);
    }

    /**
     * удаляет токен из `session`
     */
    public function unsetToken()
    {
        if (! empty($this->session[$this->sessionTokenLabel])) {
            unset($this->session[$this->sessionTokenLabel]);
        }
    }
}