<?php

namespace Mnv\Models;

use Mnv\Core\DB;
use Mnv\Core\Managers\Role;
use Mnv\Models\Contents\AbstractProduct;
use Mnv\Models\Contents\ContentInterface;
use Mnv\Models\Contents\Exceptions\NoContentException;
use Mnv\Models\Contents\Exceptions\EmptyArticleIdException;
use Mnv\Models\Contents\Exceptions\NotInsertContentException;
use Mnv\Models\Contents\Exceptions\NotUpdateContentException;
use Mnv\Models\Contents\Exceptions\NotContentApproveException;
use Mnv\Models\Contents\Exceptions\ContentNotDeletedException;

/**
 * Class Products
 * @package Mnv\Models\Contents
 */
class Products extends AbstractProduct implements ContentInterface
{

    private static $_table         = 'articles';
    private static $_table_images  = 'article_images';

    /**
     * Articles constructor.
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Получение всех статей
     *
     * @param string|null $query
     * @param string|null $status
     * @param string|null $typeContent
     * @param null $sectionIds
     * @param $orderBy
     * @param $manager
     */
    public function listContent(?string $query, ?string $status, ?string $typeContent, $sectionIds = null, $orderBy, $manager)
    {
        $this->getContents($query, $status, $typeContent, $sectionIds, $manager);
        if (!is_null($this->contents)) {
            foreach ($this->contents as $articleId => $content) {
                $this->contents[$articleId]->image = $this->getFile($content->articleId);
            }

        }
    }

    /**
     * Получение кол-во контента
     *
     * @param string|null $query
     * @param string|null $status
     * @param string|null $typeContent
     * @param null $sectionIds
     * @param $manager
     */
    public function total(?string $query, ?string $status, ?string $typeContent, $sectionIds = null, $manager)
    {
        $this->getCountContents($query, $status, $typeContent, $sectionIds, $manager);
    }

    /**
     * Редактирование и добавление контента
     *
     * @throws NoContentException
     */
    public function edit()
    {
        if (!empty($this->articleId)) {
            $this->getContent();
            if ($this->content) {
                /** получать информацию о пользователе, который создал или редактировал контент */
                $this->getManagerInfo();
                /** получение картинок контента */
                $this->getFiles();
            } else {
                throw new NoContentException();
            }
        }

    }

    /**
     * Добавление нового контента
     *
     * @param $content
     * @param $images
     * @param $manager
     * @return mixed|void
     * @throws NotInsertContentException
     */
    public function add($content, $images, $manager)
    {
        $content['addedBy']     = $manager->userId;
        $content['addedOn']     = gmdate('Y-m-d H:i:s');
        $content['modifiedBy']  = $manager->userId;
        $content['modifiedOn']  = gmdate('Y-m-d H:i:s');
        $content['publishedOn'] = adjustTime($content['publishedOn'],  true);
        $content['orderBy']     = DB::init()->connect()->table(static::$_table)->max('orderBy')->where('typeContent','=', $content['typeContent'])->getValue() + 1;

        $content['votes'] = 0;

        if ($manager->accessLevel == Role::WRITER) {
            $content['status'] = 'pending';
            $content['isFeatured'] = 0;
        }

        /* insert новую запись и возвращаем ID */
        if ($this->insertContent($content)) {
            if (!empty($images['fileId'])) $this->general($this->articleId, $images);
            if (!empty($images['gallery'])) $this->picture($this->articleId, array_keys($images['gallery']));
        } else {
            throw new NotInsertContentException();
        }

    }

    /**
     * Обновление контента
     *
     * @param $content
     * @param $images
     * @param $manager
     * @return mixed|void
     * @throws NotUpdateContentException
     */
    public function update($content, $images, $manager)
    {
        if (!empty($content['articleId'])) {
            $content['modifiedBy']   = $manager->userId;
            $content['modifiedOn']   = gmdate('Y-m-d H:i:s');
            $content['publishedOn']  = adjustTime($content['publishedOn'], true);
            if ($manager->accessLevel == Role::WRITER) {
                DB::init()->connect()->where('addedBy', $manager->userId);
                $content['status'] = 'pending';
                unset($content['isFeatured']);
            }

//            print_r($content);
            if ($this->updateContent($content)) {

                if (!empty($images['fileId'])) $this->general($this->articleId, $images);
                if (!empty($images['gallery'])) $this->picture($this->articleId, array_keys($images['gallery']));

                $this->getContent();
//            if (!empty($content->sectionIds)) $content->sectionIds = json_decode($content->sectionIds);
                /* получать информацию о пользователе, который создал или редактировал раздела */
                $this->getManagerInfo();
                /* получение картинок раздела */
                $this->getFiles();
            } else {
                throw new NotUpdateContentException();
            }

        }
    }


    /**
     * Удаление контента
     *
     * @param int $productId
     * @param $manager
     * @return mixed|void
     * @throws ContentNotDeletedException
     * @throws EmptyArticleIdException
     * @throws NoContentException
     */
    public function remove($manager)
    {
        if ( $this->articleId !== null ) {
            $this->getContent();
            if ($this->content !== null) {
                if ( !$this->removeContent($manager) ) {
                    throw new ContentNotDeletedException();
                }
            } else {
                throw new NoContentException();
            }
        } else {
            throw new EmptyArticleIdException();
        }
    }

    /**
     * Обновление статуса
     *
     * @param int $productId
     * @param $manager
     * @return mixed|void
     * @throws EmptyArticleIdException
     * @throws NoContentException
     * @throws NotContentApproveException
     */
    public function approve($manager)
    {
        if ( $this->articleId !== null ) {
            $this->getContent();
            if ($this->content !== null) {
                if (!$this->approveContent($manager)) {
                    throw new NotContentApproveException();
                }
            }else {
                throw new NoContentException();
            }
        } else {
            throw new EmptyArticleIdException();
        }
    }

    /**
     * Изменение статуса
     *
     * @throws NoContentException
     * @throws EmptyArticleIdException
     * @throws NotContentApproveException
     */
    public function status()
    {
        if ($this->articleId !== null) {
            $this->getContent();
            if ($this->content !== null) {
                $contentUpdate['status'] = ($this->content->status == 'visible') ? 'hidden' : 'visible';
                if (!$this->statusContent($contentUpdate)) {
                    throw new NotContentApproveException();
                } else {
                    $this->result = array('data' => true, 'status' => $contentUpdate['status']);
                }
            }else {
                throw new NoContentException();
            }
        } else {
            throw new EmptyArticleIdException();
        }
    }




}