<?php

namespace Mnv\Models\Users;

use Mnv\Core\DB;
use Mnv\Core\Managers\Role;
use Mnv\Models\Users\Exceptions\NoUserException;
use Mnv\Models\Users\Exceptions\EmptyUserIdException;
use Mnv\Models\Users\Exceptions\NotUpdateUserException;
use Mnv\Models\Users\Exceptions\UserNotDeletedException;
use Mnv\Models\Users\Exceptions\NotUserApproveException;
use Mnv\Models\Users\Exceptions\CannotBeRemovedDeveloperException;
use Mnv\Models\Users\Exceptions\CannotBeRemovedAdministratorException;

/**
 * Class Managers
 */
class AdminUser extends AbstractUser implements UserInterface
{

    /** @var array|string права доступа */
    public $accessLevels;
    /** @var int $totalArticle кол-во контента сланного менеджером  */
    public $totalArticle;

    /** ManagersAdmin constructor
     * @throws \SmartyException
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Получение всех менеджеров
     *
     * @param string|null $query (optional)
     * @param string|null $status (optional)
     * @param int $page
     * @param int $limit
     * @param null $columnSortOrder (optional)
     * @param null $columnName (optional)
     * @param $manager
     */
    public function getAll(?string $query, ?string $status, int $page, int $limit, $columnSortOrder = null, $columnName = null, $manager) : void
    {

        $columnIndex = $columnSortOrder[0]['column'];

        if ($columnSortOrder) {
            $orderBy = $columnName[$columnIndex]['data'] . " " . strtoupper($columnSortOrder[0]['dir']);
        } else {
            $orderBy = 'userId ASC';
        }

        $this->getUsers($query, $status, $page, $limit, 'adminUser', $orderBy, $manager);

        foreach ($this->users as $managerId => $manager) {
            $this->users[$managerId]->statusName = lang('managers:statuses:'.$manager->status);
            $this->users[$managerId]->accessLevelName = lang('managers:accessRole:'.$manager->accessLevel);
            $this->users[$managerId]->registered = langDate(adjustTime(gmdate( 'Y-m-d H:i', $manager->registered), false, 'Y-m-d H:i'));
            $this->users[$managerId]->last_login = langDate(adjustTime(gmdate( 'Y-m-d H:i', $manager->last_login), false, 'Y-m-d H:i'));

            $this->users[$managerId]->image = $this->getFile($manager->fileId);
        }
//        print_r($this->users);
    }

    /**
     * получение кол-во пользователей
     *
     * @param string|null $query
     * @param string|null $status
     * @param $manager
     */
    public function total(?string $query, ?string $status, $manager) : void
    {
        $this->getCountUsers($query, $status, 'adminUser', $manager);
    }

    /**
     * Получить кол-во пользователей при фильтрации
     *
     * @param string|null $query
     * @param $manager
     */
    public function countTableFiltered(?string $query, $manager) : void
    {
        $this->countTableFilteredUsers($query, 'adminUser', $manager);
    }

    /**
     * Редактирование
     */
    public function edit()
    {
        if (!empty($this->userId)) {
            $this->getUser();
            if ($this->user !== null) {
                $this->user->permissions = explode(',', $this->user->permissions);
                $this->user->registered = gmdate('Y-m-d H:i', $this->user->registered);
                $this->user->last_login = gmdate('Y-m-d H:i', $this->user->last_login);

                $this->user->image = $this->getFile($this->user->fileId);
                $this->user->accessLevelName = lang('managers:accessRole:' . $this->user->accessLevel);

                $totalArticle = DB::init()->connect()->table('articles')->count('*', 'count')->where('addedBy', $this->user->userId)->get();
                $this->totalArticle = $totalArticle->count;

                $this->user->notice = $this->notice();
            }
        }
    }

    private function notice(): string
    {
        $notice = '';
        if ($res =  DB::init()->connect()->table('notice')->where('userId', $this->user->userId)->get()) {
            $notice = htmlspecialchars( $res->notice, ENT_QUOTES, $this->config['charset'] );
        }

        return $notice;
    }

    /**
     * Создание нового менеджера
     *
     * @param $user
     * @param $manager
     * @return bool
     */
    public function add($user, $manager): bool
    {
        $userInsert = array_filter([
            'fullName'      => $user->fullName,
            'loginName'     => $user->loginName,
            'email'         => $user->email,
            'password'      => \password_hash($user->password, \PASSWORD_DEFAULT),
            'phone'         => $user->phone,
            'address'       => $user->address,
            'permissions'   => $user->permissions,
            'accessLevel'   => $user->accessLevel,
            'status'        => $user->status,
            'receiveEmails' => $user->receiveEmails,
            'fileId'        => $user->fileId,
            'addedBy'       => $manager->userId,
            'addedOn'       => gmdate('Y-m-d H:i:s'),
            'modifiedBy'    => $manager->userId,
            'modifiedOn'    => gmdate('Y-m-d H:i:s'),
            'registered'    => \time(),
            'last_login'    => \time(),
            'verified'      => 1,
        ]);

        if ($this->insertUser((array)$userInsert)) return true;

        return false;
    }

    /**
     * Обновление менеджера
     *
     * @param $user
     * @param $manager
     * @return bool
     * @throws NotUpdateUserException
     */
    public function update($user, $manager): bool
    {
        if (!empty($user->userId)) {
            $userUpdate = array_filter([
                'fullName'      => $user->fullName,
                'loginName'     => $user->loginName,
                'email'         => $user->email,
                'phone'         => $user->phone,
                'permissions'   => $user->permissions,
                'accessLevel'   => $user->accessLevel,
                'status'        => $user->status,
                'fileId'        => $user->fileId,
                'modifiedBy'    => $manager->userId,
                'modifiedOn'    => gmdate('Y-m-d H:i:s'),
            ]);

            if (!empty($user->newPassword)) $userUpdate['password'] = \password_hash($user->newPassword, \PASSWORD_DEFAULT);
            $userUpdate['receiveEmails'] = !empty($user->receiveEmails) ? 1 : 0;

            if ($this->updateUser($userUpdate)) {

                $this->getUser();
                $this->user->permissions = !is_null($this->user->permissions) ? explode(',', $this->user->permissions) : null;
                $this->user->image = $this->getFile($this->user->fileId);
                $this->user->accessLevelName = lang('managers:accessRole:' . $this->user->accessLevel);

                return true;
            } else {
                throw new NotUpdateUserException();
            }
        }
        return false;
    }

    /**
     * Права доступа / Access rights
     *
     * @param $accessLevelId
     * @return array
     */
    public function accessRights($accessLevelId): array
    {
        $permission = json_decode(file_read(GLOBAL_ROOT."/admin/menu/menu.json"), true);
        $result = array();

        foreach ($permission as $key => $item) {
            if (!in_array($accessLevelId, $item['accessLevel']))  continue;
            $result[$key] = $item;
            $child = array();
            foreach ($result[$key]['children'] as $childrenKey => $children) {
                if (!in_array($accessLevelId, $children['accessLevel'])) continue;
                $child[$childrenKey] = $children;
            }
            unset( $result[$key]['children']);

            $result[$key]['children'] = $child;
        }

        return $result;
    }


    /**
     * Удаление пользователя
     *
     * @throws NoUserException
     * @throws EmptyUserIdException
     * @throws UserNotDeletedException
     * @throws CannotBeRemovedDeveloperException
     * @throws CannotBeRemovedAdministratorException
     */
    public function remove()
    {
        if ( $this->userId !== null ) {
            $this->getUser();
            if ($this->user !== null) {
                if ( $this->user->accessLevel == Role::DEVELOPER ) {
                    if ( !$this->getUserByRole(Role::DEVELOPER) ) {
                        throw new CannotBeRemovedDeveloperException();
                    }
                } else if ( $this->user->accessLevel == Role::ADMIN ) {
                    if ( !$this->getUserByRole(Role::ADMIN) ) {
                        throw new CannotBeRemovedAdministratorException();
                    }
                }
                if ( !$this->removeUser() ) {
                    throw new UserNotDeletedException();
                }
            } else {
                throw new NoUserException();
            }
        } else {
            throw new EmptyUserIdException();
        }
    }

    /**
     * Обновление статуса
     *
     * @throws NoUserException
     * @throws EmptyUserIdException
     * @throws NotUserApproveException
     */
    public function approve()
    {
        if ( $this->userId !== null ) {
            $this->getUser();
            if ($this->user !== null) {
                if (!$this->approveUser()) {
                    throw new NotUserApproveException();
                }
            }else {
                throw new NoUserException();
            }
        } else {
            throw new EmptyUserIdException();
        }
    }



}