<?php


namespace Mnv\Modules\Telegram\Emoji;

/**
 * Class Emojify
 * @package Mnv\Modules\Telegram\Emoji
 */
class Emojify
{
    /**
     * @var Emojify Ссылка на *Singleton* экземпляр этого класса
     */
    private static $instance;

    /**
     * Путь к файлу, содержащему карту эмодзи.
     *
     * @var string
     */
    const DEFAULT_EMOJI_MAP_FILE = '/emoji.json';

    /**
     * Путь к файлу, содержащему карту эмодзи.
     *
     * @var string
     */
    protected $emojiMapFile;

    /**
     * Массив, сопоставляющий слова с эмодзи.
     *
     * @var array
     */
    protected $emojiMap;

    /**
     * Массив, отображающий эмодзи обратно в слова.
     *
     * @var array
     */
    protected $wordMap;

    /**
     * Защищенный конструктор Emojify для предотвращения создания нового экземпляра
     * *Singleton* с помощью оператора `new` извне этого класса.
     *
     * @throws TelegramEmojiMapFileNotFoundException
     */
    protected function __construct()
    {
        $this->setupEmojiMaps();
    }

    /**
     *Возвращает экземпляр *Singleton* этого класса.
     *
     * @return Emojify Экземпляр *Singleton*.
     */
    public static function getInstance()
    {
        if (null === static::$instance) {
            static::$instance = new static();
        }

        return static::$instance;
    }

    /**
     * Задайте путь к файлу карты Emoji.
     *
     * @param string $emojiMapFile
     * @return Emojify
     * @throws TelegramEmojiMapFileNotFoundException
     */
    public function setEmojiMapFile($emojiMapFile)
    {
        $this->emojiMapFile = $emojiMapFile;
        $this->setupEmojiMaps();

        return $this;
    }

    /**
     * Переведите слово в эмодзи.
     *
     * @param $text
     * @return mixed
     */
    public function toEmoji($text)
    {
        return $this->replace($text, $this->emojiMap);
    }

    /**
     * Псевдоним toEmoji().
     *
     * @param $text
     * @return mixed
     */
    public static function text($text)
    {
        return self::getInstance()->toEmoji($text);
    }

    /**
     * Переведите эмодзи в слово.
     *
     * @param $text
     * @return mixed
     */
    public function toWord($text)
    {
        return $this->replace($text, $this->wordMap, true);
    }

    /**
     * Псевдоним toWord().
     *
     * @param $text
     * @return mixed
     */
    public static function translate($text)
    {
        return self::getInstance()->toWord($text);
    }

    /**
     * Заменять.
     *
     * @param    $line
     * @param    $replace
     * @param  bool  $toWord
     * @param  string  $delimiter
     * @return mixed
     */
    protected function replace($line, $replace, $toWord = false, $delimiter = ':')
    {
        if ($toWord) {
            return $this->emojiToWordReplace($line, $replace, $delimiter);
        }

        return $this->wordToEmojiReplace($line, $replace, $delimiter);
    }

    /**
     * Находит слова, заключенные в разделитель, и преобразует их в соответствующий символ эмодзи.
     *
     * @param $line
     * @param $replace
     * @param $delimiter
     * @return mixed
     */
    protected function wordToEmojiReplace($line, $replace, $delimiter)
    {
        return $replace[$line];
//        foreach ($replace as $key => $value) {
//            $line = str_replace($delimiter.$key.$delimiter, $value, $line);
//        }
//
//        return $line;
    }

    /**
     * Находит эмодзи и заменяет их текстом, заключенным в разделитель.
     *
     * @param $line
     * @param $replace
     * @param $delimiter
     * @return mixed
     */
    protected function emojiToWordReplace($line, $replace, $delimiter)
    {
        foreach ($replace as $key => $value) {
            $line = str_replace($key, $delimiter.$value.$delimiter, $line);
        }

        return $line;
    }

    /**
     * Получите Массив карт Эмодзи.
     *
     * @return array
     *
     * @throws TelegramEmojiMapFileNotFoundException
     */
    protected function getEmojiMap()
    {
        if (!isset($this->emojiMapFile)) {
            $this->emojiMapFile = realpath(__DIR__.self::DEFAULT_EMOJI_MAP_FILE);
        }

        if (! file_exists($this->emojiMapFile)) {
            throw new TelegramEmojiMapFileNotFoundException();
        }

        return json_decode(file_get_contents($this->emojiMapFile), true);
    }

    /**
     * Настройка карт Эмодзи.
     *
     * @throws TelegramEmojiMapFileNotFoundException
     */
    protected function setupEmojiMaps()
    {
        $this->emojiMap = $this->getEmojiMap();
        $this->wordMap = array_flip($this->emojiMap);
    }

    /**
     * Выдает исключение, когда пользователь пытается клонировать экземпляр *Singleton*.
     *
     * @throws \LogicException always
     */
    public function __clone()
    {
        throw new \LogicException('The Emojify helper cannot be cloned');
    }

    /**
     * Выдает исключение, когда пользователь пытается отменить сериализацию экземпляра *Singleton*.
     *
     * @throws \LogicException always
     */
    public function __wakeup()
    {
        throw new \LogicException('The Emojify helper cannot be serialised');
    }
}