<?php
namespace Mnv\Modules\Telegram\Yandex;

use Mnv\Modules\Telegram\Yandex\Exception\CurlError;
use Mnv\Modules\Telegram\Yandex\Exception\MapsError;
use Mnv\Modules\Telegram\Yandex\Exception\ServerError;
use Mnv\Modules\Telegram\Yandex\Exception\YandexException;

/**
 * Class YandexApi
 * @package Mnv\Modules\Telegram\Yandex
 * @see https://yandex.ru/dev/maps/geosearch/doc/concepts/about.html
 */
class YandexApi
{
    /** русский (по умолчанию) */
    const LANG_RU = 'ru-RU';
    /** украинский */
    const LANG_UA = 'uk-UA';
    /** белорусский */
    const LANG_BY = 'be-BY';
    /** американский английский */
    const LANG_US = 'en-US';
    /** британский английский */
    const LANG_BR = 'en-BR';
    /** турецкий (только для карты Турции) */
    const LANG_TR = 'tr-TR';

    /** @var string Версия используемого api */
    protected string $_version = 'v1';

    /** @var array $_filters*/
    protected array $_filters = array();

    /** @var Response|null */
    protected ?Response $_response;

    /**
     * @param string $token
     * @param null|string $version
     * @throws YandexException
     */
    public function __construct(string $token, string $type = null, string $version = null)
    {
        if (empty($token)) throw new YandexException("Key is required");


        if (!empty($version)) {
            $this->_version = $version;
        }

        $this->clear($token, $type);
    }

    /**
     * @param array $options Curl options
     * @return $this
     * @throws YandexException
     * @throws CurlError
     * @throws ServerError
     * @throws MapsError
     */
    public function load(array $options = [])
    {
        $apiUrl = sprintf('https://search-maps.yandex.ru/%s/?%s', $this->_version, http_build_query($this->_filters));
        $curl = curl_init($apiUrl);
        $options += array(
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_HTTPGET => 1,
            CURLOPT_FOLLOWLOCATION => 1,
        );
        curl_setopt_array($curl, $options);
        $data = curl_exec($curl);
        $code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        if (curl_errno($curl)) {
            $error = curl_error($curl);
            curl_close($curl);
            throw new CurlError($error);
        }
        curl_close($curl);
        if (in_array($code, array(500, 502))) {
            $msg = strip_tags($data);
            throw new ServerError(trim($msg), $code);
        }
        $data = json_decode($data, true);
        if (empty($data)) {
            $msg = sprintf('Can\'t load data by url: %s', $apiUrl);
            throw new YandexException($msg);
        }
        if (!empty($data['error'])) {
            throw new YandexException($data['error'] . ": ".  $data['message'], $data['statusCode']);
        }

//        print_r($data);
        $this->_response = new Response($data);
        return $this;
    }

    /**
     * @return Response
     */
    public function getResponse(): ?Response
    {
        return $this->_response;
    }

    /**
     * Очистка фильтров гео-кодирования
     *
     * @param string $token
     * @param string|null $type
     * @return $this
     */
    public function clear(string $token, ?string $type): YandexApi
    {
        $this->_filters = array('format' => 'json');
        // указываем явно значения по-умолчанию
        $this->setToken($token)->setType($type)->setLang(self::LANG_RU)->setOffset(0)->setLimit(30);
        $this->_response = null;

        return $this;
    }

    /**
     * Типы возвращаемых результатов. Возможные значения: geo — топонимы; biz — организации;
     * @param string|null $type
     * @return self
     */
    public function setType(?string $type): YandexApi
    {
        if (!empty($type)) $this->_filters['type'] = $type;
        return $this;
    }

    /**
     * центр области поиска ll=37.618920,55.756994
     * @param float $longitude Долгота в градусах
     * @param float $latitude Широта в градусах
     * @return self
     */
    public function setLL(float $longitude, float $latitude): YandexApi
    {
        $this->_filters['ll'] = sprintf('%F,%F', $longitude, $latitude);
        return $this;
    }

    /**
     * размеры области поиска spn=0.552069,0.400552
     * @param float $longitude Долгота в градусах
     * @param float $latitude Широта в градусах
     * @return self
     */
    public function setSPN(float $longitude, float $latitude): YandexApi
    {
        $this->_filters['spn'] = sprintf('%F,%F', $longitude, $latitude);
        return $this;
    }

    /**
     * координаты области поиска bbox=36.83,55.67~38.24,55.91
     * @param float $longitude Долгота в градусах
     * @param float $latitude Широта в градусах
     * @return self
     */
    public function setBBOX(float $longitude, float $latitude): YandexApi
    {
        $this->_filters['spn'] = sprintf('%F,%F', $longitude, $latitude);
        return $this;
    }

    /**
     * Гео-кодирование по запросу (адрес/координаты)
     * @param string $query
     * @return self
     */
    public function setQuery(string $query): YandexApi
    {
        $this->_filters['text'] = (string)$query;
        return $this;
    }


    /**
     * Максимальное количество возвращаемых объектов (по-умолчанию 10)
     * @param int $limit
     * @return self
     */
    public function setLimit(int $limit): YandexApi
    {
        $this->_filters['results'] = $limit;
        return $this;
    }

    /**
     * Количество объектов в ответе (начиная с первого), которое необходимо пропустить
     * @param int $offset
     * @return self
     */
    public function setOffset(int $offset): YandexApi
    {
        $this->_filters['skip'] = $offset;
        return $this;
    }

    /**
     * Предпочитаемый язык описания объектов
     * @param string $lang
     * @return self
     */
    public function setLang(string $lang): YandexApi
    {
        $this->_filters['lang'] = (string)$lang;
        return $this;
    }

    /**
     * Ключ API Яндекс.Карт
     *
     * @param string $token
     * @return self
     */
    public function setToken(string $token): YandexApi
    {
        $this->_filters['apikey'] = $token;
        return $this;
    }
}
